/*
 * Copyright IBM Corp. and others 2008
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package com.ibm.java.lang.management.internal;

import java.io.IOException;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.MemoryType;
import java.lang.management.RuntimeMXBean;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularDataSupport;
import javax.management.openmbean.TabularType;

/**
 * Support methods for com.ibm.lang.management classes.
 * Provides utility methods to other packages (hence, specified as public),
 * such as JLM classes, but not meant to be published as an API.
 */
@SuppressWarnings("javadoc")
public final class ManagementUtils {

	/*
	 * String representation for the type NotificationEmitter that must
	 * be implemented by a bean that emits notifications.
	 */
	private static final String NOTIFICATION_EMITTER_TYPE = "javax.management.NotificationEmitter"; //$NON-NLS-1$

	/* Set to true, if we are running on a Unix or Unix like operating
	 * system; false, otherwise.
	 */
	private static final boolean isUnix;

	/**
	 * System property setting used to decide if non-fatal exceptions should be
	 * written out to console.
	 */
	public static final boolean VERBOSE_MODE;

	static {
		Properties properties = com.ibm.oti.vm.VM.internalGetProperties();
		String thisOs = properties.getProperty("os.name"); //$NON-NLS-1$

		isUnix = "aix".equalsIgnoreCase(thisOs) //$NON-NLS-1$
				|| "linux".equalsIgnoreCase(thisOs) //$NON-NLS-1$
				|| "mac os x".equalsIgnoreCase(thisOs) //$NON-NLS-1$
				|| "z/OS".equalsIgnoreCase(thisOs); //$NON-NLS-1$

		VERBOSE_MODE = properties.getProperty("com.ibm.lang.management.verbose") != null; //$NON-NLS-1$
	}

	/**
	 * Throws an {@link IllegalArgumentException} if the {@link CompositeData}
	 * argument <code>cd</code> contains attributes that are not of the exact
	 * types specified in the <code>expectedTypes</code> argument. The
	 * attribute types of <code>cd</code> must also match the order of types
	 * in <code>expectedTypes</code>.
	 *
	 * @param cd
	 *            a <code>CompositeData</code> object
	 * @param expectedNames
	 *            an array of expected attribute names
	 * @param expectedTypes
	 *            an array of type names
	 */
	public static void verifyFieldTypes(CompositeData cd, String[] expectedNames, String[] expectedTypes) {
		Object[] allVals = cd.getAll(expectedNames);
		// Check that the number of elements match
		if (allVals.length != expectedTypes.length) {
			// K05E8 = CompositeData does not contain the expected number of attributes.
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K05E8")); //$NON-NLS-1$
		}

		// Type of corresponding elements must be the same
		for (int i = 0; i < allVals.length; ++i) {
			Object actualVal = allVals[i];
			// It is permissible that a value in the CompositeData object is
			// null in which case we cannot test its type. Move on.
			if (actualVal == null) {
				continue;
			}
			String actualType = actualVal.getClass().getName();
			String expectedType = expectedTypes[i];
			if (actualType.equals(expectedType)) {
				continue;
			}
			// Handle CompositeData and CompositeDataSupport
			if (expectedType.equals(CompositeData.class.getName())) {
				if (actualVal instanceof CompositeData) {
					continue;
				}
			} else if (expectedType.equals(TabularData.class.getName())) {
				if (actualVal instanceof TabularData) {
					continue;
				}
			}
			// K05E9 = CompositeData contains an attribute of unexpected type. Expected {0}, found {1}.
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K05E9", //$NON-NLS-1$
					expectedType, actualType));
		}
	}

	/**
	 * Throws an {@link IllegalArgumentException} if the {@link CompositeData}
	 * argument <code>cd</code> does not have any of the attributes named in
	 * the <code>expected</code> array of strings.
	 *
	 * @param cd
	 *            a <code>CompositeData</code> object
	 * @param expected
	 *            an array of attribute names expected in <code>cd</code>.
	 */
	public static void verifyFieldNames(CompositeData cd, String[] expected) {
		for (String expect : expected) {
			if (!cd.containsKey(expect)) {
				// K05EA = CompositeData object does not contain expected key : {0}.
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K05EA", expect)); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Throws an {@link IllegalArgumentException} if the {@link CompositeData}
	 * argument <code>cd</code> does not have the number of attributes
	 * specified in <code>i</code>.
	 *
	 * @param cd
	 *            a <code>CompositeData</code> object
	 * @param i
	 *            the number of expected attributes in <code>cd</code>
	 */
	public static void verifyFieldNumber(CompositeData cd, int i) {
		if (cd == null) {
			// K05EB = Null CompositeData
			throw new NullPointerException(com.ibm.oti.util.Msg.getString("K05EB")); //$NON-NLS-1$
		}
		if (cd.values().size() != i) {
			// K05EC = CompositeData object does not have the expected number of attributes
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K05EC")); //$NON-NLS-1$
		}
	}

	/**
	 * Convenience method to converts an array of <code>String</code> to a
	 * <code>List&lt;String&gt;</code>.
	 *
	 * @param data
	 *            an array of <code>String</code>
	 * @return a new <code>List&lt;String&gt;</code>
	 */
	public static List<String> convertStringArrayToList(String[] data) {
		List<String> result = new ArrayList<>(data.length);

		for (String string : data) {
			result.add(string);
		}

		return result;
	}

	/**
	 * Receives an instance of a {@link TabularData} whose data is wrapping a
	 * <code>Map</code> and returns a new instance of <code>Map</code>
	 * containing the input information.
	 *
	 * @param data
	 *            an instance of <code>TabularData</code> that may be mapped
	 *            to a <code>Map</code>.
	 * @return a new {@link Map} containing the information originally wrapped
	 *         in the <code>data</code> input.
	 * @throws IllegalArgumentException
	 *             if <code>data</code> has a <code>CompositeType</code>
	 *             that does not contain exactly two items (i.e. a key and a
	 *             value).
	 */
	public static Object convertTabularDataToMap(TabularData data) {
		// Bail out early on null input.
		if (data == null) {
			return null;
		}

		Set<String> cdKeySet = data.getTabularType().getRowType().keySet();

		// The key set for the CompositeData instances comprising each row
		// must contain only two elements.
		if (cdKeySet.size() != 2) {
			// K05ED = TabularData's row type is not a CompositeType with two items.
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K05ED")); //$NON-NLS-1$
		}

		String[] keys = cdKeySet.toArray(new String[2]);

		@SuppressWarnings("unchecked")
		Collection<CompositeData> rows = (Collection<CompositeData>) data.values();
		// HashMap.DEFAULT_LOAD_FACTOR is 0.75
		Map<Object, Object> result = new HashMap<>(rows.size() * 4 / 3);

		for (CompositeData rowCD : rows) {
			result.put(rowCD.get(keys[0]), rowCD.get(keys[1]));
		}

		return result;
	}

	/**
	 * Return a new instance of type <code>T</code> from the supplied
	 * {@link CompositeData} object whose type maps to <code>T</code>.
	 *
	 * @param <T>
	 *            the type of object wrapped by the <code>CompositeData</code>.
	 * @param data
	 *            an instance of <code>CompositeData</code> that maps to an
	 *            instance of <code>T</code>
	 * @param realClass
	 *            the {@link Class} object for type <code>T</code>
	 * @return a new instance of <code>T</code>
	 * @throws NoSuchMethodException
	 * @throws SecurityException
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	private static <T> T convertFromCompositeData(CompositeData data, Class<T> realClass) throws SecurityException,
			NoSuchMethodException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		// Bail out early on null input.
		if (data == null) {
			return null;
		}

		// See if the realClass has a static 'from' method that takes a
		// CompositeData and returns a new instance of T.
		Method forMethod = realClass.getMethod("from", CompositeData.class); //$NON-NLS-1$

		return realClass.cast(forMethod.invoke(null, data));
	}

	/**
	 * Receive data of the type specified in <code>openClass</code> and return
	 * it in an instance of the type specified in <code>realClass</code>.
	 *
	 * @param <T>
	 *
	 * @param data
	 *            an instance of the type named <code>openTypeName</code>
	 * @param openClass
	 * @param realClass
	 * @return a new instance of the type <code>realTypeName</code> containing
	 *         all the state in the input <code>data</code> object.
	 * @throws ClassNotFoundException
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 * @throws InvocationTargetException
	 * @throws NoSuchMethodException
	 * @throws IllegalArgumentException
	 * @throws SecurityException
	 */
	@SuppressWarnings("unchecked")
	public static <T> T convertFromOpenType(Object data, Class<?> openClass,
			Class<T> realClass) throws ClassNotFoundException,
			InstantiationException, IllegalAccessException, SecurityException,
			IllegalArgumentException, NoSuchMethodException,
			InvocationTargetException {
		// Bail out early on null input.
		if (data == null) {
			return null;
		}

		T result = null;

		if (openClass.isArray() && realClass.isArray()) {
			Class<?> openElementClass = openClass.getComponentType();
			Class<?> realElementClass = realClass.getComponentType();
			Object[] dataArray = (Object[]) data;
			int length = dataArray.length;

			result = (T) Array.newInstance(realElementClass, length);

			for (int i = 0; i < length; ++i) {
				Array.set(result, i, convertFromOpenType(dataArray[i], openElementClass, realElementClass));
			}
		} else if (openClass.equals(CompositeData.class)) {
			result = ManagementUtils.convertFromCompositeData((CompositeData) data, realClass);
		} else if (openClass.equals(TabularData.class)) {
			if (realClass.equals(Map.class)) {
				result = (T) ManagementUtils.convertTabularDataToMap((TabularData) data);
			}
		} else if (openClass.equals(String[].class)) {
			if (realClass.equals(List.class)) {
				result = (T) ManagementUtils.convertStringArrayToList((String[]) data);
			}
		} else if (openClass.equals(String.class)) {
			if (realClass.equals(MemoryType.class)) {
				result = (T) ManagementUtils.convertStringToMemoryType((String) data);
			}
		}

		return result;
	}

	/**
	 * Convenience method that receives a string representation of a
	 * <code>MemoryType</code> instance and returns the actual
	 * <code>MemoryType</code> that corresponds to that value.
	 *
	 * @param data
	 *            a string
	 * @return if <code>data</code> can be used to obtain an instance of
	 *         <code>MemoryType</code> then a <code>MemoryType</code>,
	 *         otherwise <code>null</code>.
	 */
	private static MemoryType convertStringToMemoryType(String data) {
		MemoryType result = null;

		try {
			result = MemoryType.valueOf(data);
		} catch (IllegalArgumentException e) {
			if (ManagementUtils.VERBOSE_MODE) {
				e.printStackTrace(System.err);
			}
		}

		return result;
	}

	/**
	 * @param propsMap
	 *            a <code>Map&lt;String, String%gt;</code> of the system
	 *            properties.
	 * @return the system properties (e.g. as obtained from
	 *         {@link RuntimeMXBean#getSystemProperties()}) wrapped in a
	 *         {@link TabularData}.
	 */
	public static TabularData toSystemPropertiesTabularData(Map<String, String> propsMap) {
		// Bail out early on null input.
		if (propsMap == null) {
			return null;
		}

		TabularData result = null;
		try {
			// Obtain the row type for the TabularType
			String[] rtItemNames = { "key", "value" }; //$NON-NLS-1$ //$NON-NLS-2$
			String[] rtItemDescs = { "key", "value" }; //$NON-NLS-1$ //$NON-NLS-2$
			OpenType<?>[] rtItemTypes = { SimpleType.STRING, SimpleType.STRING };

			CompositeType rowType = new CompositeType(
					propsMap.getClass().getName(),
					propsMap.getClass().getName(),
					rtItemNames,
					rtItemDescs,
					rtItemTypes);

			// Obtain the required TabularType
			TabularType sysPropsType = new TabularType(
					propsMap.getClass().getName(),
					propsMap.getClass().getName(),
					rowType,
					new String[] { "key" }); //$NON-NLS-1$

			// Create an empty TabularData
			result = new TabularDataSupport(sysPropsType);

			// Take each entry out of the input propsMap, put it into a new
			// instance of CompositeData and put into the TabularType
			for (Entry<String, String> entry : propsMap.entrySet()) {
				String propKey = entry.getKey();
				String propVal = entry.getValue();

				result.put(new CompositeDataSupport(rowType, rtItemNames, new String[] { propKey, propVal }));
			}
		} catch (OpenDataException e) {
			if (ManagementUtils.VERBOSE_MODE) {
				e.printStackTrace(System.err);
			}
			result = null;
		}

		return result;
	}

	/**
	 * Convenience method to determine if the <code>wrapper</code>
	 * <code>Class</code>
	 * object is really the wrapper class for the
	 * <code>primitive</code> <code>Class</code> object.
	 *
	 * @param wrapper
	 * @param primitive
	 * @return <code>true</code> if the <code>wrapper</code> class is the
	 *         wrapper class for <code>primitive</code>. Otherwise
	 *         <code>false</code>.
	 */
	public static boolean isWrapperClass(Class<?> wrapper, Class<?> primitive) {
		boolean result = true;
		if (primitive.equals(boolean.class) && !wrapper.equals(Boolean.class)) {
			result = false;
		} else if (primitive.equals(char.class) && !wrapper.equals(Character.class)) {
			result = false;
		} else if (primitive.equals(byte.class) && !wrapper.equals(Byte.class)) {
			result = false;
		} else if (primitive.equals(short.class) && !wrapper.equals(Short.class)) {
			result = false;
		} else if (primitive.equals(int.class) && !wrapper.equals(Integer.class)) {
			result = false;
		} else if (primitive.equals(long.class) && !wrapper.equals(Long.class)) {
			result = false;
		} else if (primitive.equals(float.class) && !wrapper.equals(Float.class)) {
			result = false;
		} else if (primitive.equals(double.class) && !wrapper.equals(Double.class)) {
			result = false;
		}

		return result;
	}

	/**
	 * Query the server via the given connection whether mxbeanObjectName
	 * is the name of a bean that is a notification emitter.
	 *
	 * @param connection
	 * @param mxbeanObjectName
	 * @return
	 * @throws IOException
	 * @throws IllegalArgumentException
	 */
	public static boolean isANotificationEmitter(MBeanServerConnection connection, ObjectName mxbeanObjectName)
			throws IOException, IllegalArgumentException {
		boolean result = false;

		try {
			result = connection.isInstanceOf(mxbeanObjectName, NOTIFICATION_EMITTER_TYPE);
		} catch (InstanceNotFoundException e) {
			// a non-existent bean is not an emitter
			result = false;
		}

		return result;
	}

	/**
	 * Helper function that tells whether we are running on Unix or not.
	 * @return Returns true if we are running on Unix; false, otherwise.
	 */
	public static boolean isRunningOnUnix() {
		return isUnix;
	}

	/**
	 * Convenience method to create an ObjectName from the specified string.
	 *
	 * @param name
	 * @return an ObjectName corresponding to the specified string.
	 */
	public static ObjectName createObjectName(String name) {
		try {
			return ObjectName.getInstance(name);
		} catch (MalformedObjectNameException e) {
			if (ManagementUtils.VERBOSE_MODE) {
				e.printStackTrace();
			}
			return null;
		}
	}

	/**
	 * Convenience method to create an ObjectName with the specified domain and name property.
	 *
	 * @param domain
	 * @param name
	 * @return an ObjectName with the specified domain and name property.
	 */
	public static ObjectName createObjectName(String domain, String name) {
		try {
			return ObjectName.getInstance(domain + ",name=" + name); //$NON-NLS-1$
		} catch (MalformedObjectNameException e) {
			if (ManagementUtils.VERBOSE_MODE) {
				e.printStackTrace();
			}
			return null;
		}
	}

	/**
	 * The prefix for all <code>ObjectName</code> strings which represent a
	 * {@link GarbageCollectorMXBean}. The unique <code>ObjectName</code> for
	 * a <code>GarbageCollectorMXBean</code> can be formed by adding
	 * &quot;,name=<i>collector name</i>&quot; to this constant.
	 */
	public static final String BUFFERPOOL_MXBEAN_DOMAIN_TYPE = "java.nio:type=BufferPool"; //$NON-NLS-1$

}
