/*
 * Copyright IBM Corp. and others 2022
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package openj9.internal.foreign.abi;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodType;

import jdk.incubator.foreign.Addressable;
import jdk.incubator.foreign.MemoryAddress;
import jdk.incubator.foreign.MemorySegment;
import jdk.incubator.foreign.ResourceScope;

/**
 * The meta data consists of the callee MH and a cache of 2 elements for MH resolution,
 * which are used to generate an upcall handler to the requested java method.
 */
@SuppressWarnings("nls")
final class UpcallMHMetaData {

	/* The target method handle intended for upcall which is placed on the java stack
	 * in native2InterpJavaUpcallImpl for call-in so as to invoke the method handle.
	 */
	private final MethodHandle calleeMH;
	/* The cache array stores MemberName and appendix which are populated
	 * by MethodHandleResolver.ffiCallLinkCallerMethod().
	 */
	private Object[] invokeCache;
	/* The argument array stores the memory specific argument (struct/pointer) object
	 * being allocated in native for upcall to stop GC from updating the previously
	 * allocated argument reference when allocating the next argument.
	 */
	private Object[] nativeArgArray;

	private ResourceScope scope;

	private static synchronized native void resolveUpcallDataInfo();

	static {
		/* Resolve the methods/fields (offset in the JCL constant pool of VM) related to the metadata
		 * given the generated macros from vmconstantpool.xml depend on their offsets to access the
		 * corresponding methods/fields in the process of the upcall.
		 */
		resolveUpcallDataInfo();
	}

	UpcallMHMetaData(MethodHandle targetHandle, int nativeArgCount, ResourceScope scope)
	{
		calleeMH = targetHandle;
		nativeArgArray = new Object[nativeArgCount];
		/* Only hold the confined scope (owned by the current thread) or
		 * the global scope (created once and shared by any thread) will
		 * be used to construct a MemorySegment object for argument in
		 * the upcall dispatcher.
		 */
		this.scope = ((scope != null) && (scope.ownerThread() != null)) ? scope : ResourceScope.globalScope();
	}

	/* Determine whether the memory segment (JDK20+) or the memory address (JDK17)
	 * of the passed-in/returned pointer is allocated in the native memory or not.
	 *
	 * Note:
	 * The method is shared in downcall and upcall.
	 */
	static void validateNativeArgRetAddrOfPtr(MemoryAddress argRetAddrOfPtr) {
		if (argRetAddrOfPtr == null) {
			throw new NullPointerException("A null pointer is not allowed.");
		}
		if (!argRetAddrOfPtr.isNative()) {
			throw new IllegalArgumentException("A heap address is not allowed: " + argRetAddrOfPtr);
		}
	}

	/* Determine whether the passed-in/returned segment is allocated in the native memory or not
	 * and return the segment if valid; otherwise, return the newly allocated native segment with
	 * all values copied from the heap segment.
	 *
	 * Note:
	 * The method is shared in downcall and upcall.
	 */
	static MemorySegment getNativeArgRetSegment(MemorySegment argRetSegment) {
		if (argRetSegment == null) {
			throw new NullPointerException("A null value is not allowed for struct.");
		}
		MemorySegment nativeSegment = argRetSegment;

		/* Copy all values in the heap segment to a newly allocated native segment
		 * given a heap segment with a zero address can't be accessed in native.
		 */
		if (!argRetSegment.isNative()) {
			nativeSegment = MemorySegment.allocateNative(argRetSegment.byteSize(), ResourceScope.globalScope());
			nativeSegment.copyFrom(argRetSegment);
		}

		return nativeSegment;
	}
}
