/*
 * Copyright IBM Corp. and others 2007
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package java.lang;

import java.lang.annotation.Annotation;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodType;
import jdk.internal.misc.Unsafe;
import java.lang.StringConcatHelper;
import java.lang.reflect.Constructor;
import java.lang.reflect.Executable;
import java.lang.reflect.Method;
import java.util.Map;

import com.ibm.oti.reflect.AnnotationParser;
import com.ibm.oti.reflect.TypeAnnotationParser;

import java.io.InputStream;
import java.io.IOException;
import java.lang.module.ModuleDescriptor;
import java.net.URL;
import java.net.URI;
import java.security.ProtectionDomain;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import java.nio.charset.Charset;
import java.nio.charset.CharacterCodingException;
import jdk.internal.access.JavaLangAccess;
import jdk.internal.module.ServicesCatalog;
import jdk.internal.reflect.ConstantPool;
import java.util.concurrent.Callable;
import jdk.internal.misc.CarrierThreadLocal;
import jdk.internal.vm.Continuation;
import jdk.internal.vm.ContinuationScope;
import jdk.internal.vm.StackableScope;
import jdk.internal.vm.ThreadContainer;
import sun.nio.ch.Interruptible;
import sun.reflect.annotation.AnnotationType;

/**
 * Helper class to allow privileged access to classes
 * from outside the java.lang package. The sun.misc.SharedSecrets class
 * uses an instance of this class to access private java.lang members.
 */
final class Access implements JavaLangAccess {

	/**
	 * Get the AnnotationType instance corresponding to this class.
	 * (This method only applies to annotation types.)
	 */
	@Override
	public AnnotationType getAnnotationType(java.lang.Class<?> arg0) {
		return arg0.getAnnotationType();
	}

	/** Return the constant pool for a class. */
	@Override
	public native ConstantPool getConstantPool(java.lang.Class<?> arg0);

	/**
	 * Return the constant pool for a class. This will call the exact same
	 * native as 'getConstantPool(java.lang.Class<?> arg0)'. We need this
	 * version so we can call it with InternRamClass instead of j.l.Class
	 * (which is required by JAVALANGACCESS).
	 */
	public static native ConstantPool getConstantPool(Object arg0);

	/**
	 * Returns the elements of an enum class or null if the
	 * Class object does not represent an enum type;
	 * the result is uncloned, cached, and shared by all callers.
	 */
	@Override
	public <E extends Enum<E>> E[] getEnumConstantsShared(java.lang.Class<E> arg0) {
		return arg0.getEnumConstantsShared();
	}

	@Override
	public void registerShutdownHook(int arg0, boolean arg1, Runnable arg2) {
		Shutdown.add(arg0, arg1, arg2);
	}

	/**
	 * Compare-And-Swap the AnnotationType instance corresponding to this class.
	 * (This method only applies to annotation types.)
	 */
	@Override
	public boolean casAnnotationType(final Class<?> clazz, AnnotationType oldType, AnnotationType newType) {
		return clazz.casAnnotationType(oldType, newType);
	}

	/**
	 * @param clazz Class object
	 * @return the array of bytes for the Annotations for clazz, or null if clazz is null
	 */
	@Override
	public byte[] getRawClassAnnotations(Class<?> clazz) {
		return AnnotationParser.getAnnotationsData(clazz);
	}

	@SuppressWarnings("removal")
	@Override
	public Thread newThreadWithAcc(Runnable runnable, java.security.AccessControlContext acc) {
		return new Thread(runnable, acc);
	}

	@Override
	public Map<java.lang.Class<? extends Annotation>, Annotation> getDeclaredAnnotationMap(
			java.lang.Class<?> arg0) {
		throw new Error("getDeclaredAnnotationMap unimplemented"); //$NON-NLS-1$
	}

	@Override
	public byte[] getRawClassTypeAnnotations(java.lang.Class<?> clazz) {
		return TypeAnnotationParser.getTypeAnnotationsData(clazz);
	}

	@Override
	public byte[] getRawExecutableTypeAnnotations(Executable exec) {
		byte[] result = null;
		if (Method.class.isInstance(exec)) {
			Method jlrMethod = (Method) exec;
			result = TypeAnnotationParser.getTypeAnnotationsData(jlrMethod);
		} else if (Constructor.class.isInstance(exec)) {
			Constructor<?> jlrConstructor = (Constructor<?>) exec;
			result = TypeAnnotationParser.getTypeAnnotationsData(jlrConstructor);
		} else {
			throw new Error("getRawExecutableTypeAnnotations not defined for "+exec.getClass().getName()); //$NON-NLS-1$
		}
		return result;
	}

	@Override
	public void invokeFinalize(java.lang.Object arg0)
			throws java.lang.Throwable {
		/*
		 * Not required for J9.
		 */
		throw new Error("invokeFinalize unimplemented"); //$NON-NLS-1$
	}

	@Override
	public String fastUUID(long param1, long param2) {
		return Long.fastUUID(param1, param2);
	}

	@Override
	public Package definePackage(ClassLoader classLoader, String name, Module module) {
		if (null == classLoader) {
			classLoader = ClassLoader.bootstrapClassLoader;
		}
		return classLoader.definePackage(name, module);
	}

	@Override
	public ConcurrentHashMap<?, ?> createOrGetClassLoaderValueMap(
			java.lang.ClassLoader classLoader) {
		return classLoader.createOrGetClassLoaderValueMap();
	}

	@SuppressWarnings("removal")
	@Override
	public void invalidatePackageAccessCache() {
		SecurityManager.invalidatePackageAccessCache();
	}

	@Override
	public Class<?> defineClass(ClassLoader classLoader, String className, byte[] classRep, ProtectionDomain protectionDomain, String str) {
		ClassLoader targetClassLoader = (null == classLoader) ? ClassLoader.bootstrapClassLoader : classLoader;
		return targetClassLoader.defineClassInternal(className, classRep, 0, classRep.length, protectionDomain, true /* allowNullProtectionDomain */);
	}

	@Override
	public Stream<ModuleLayer> layers(ModuleLayer ml) {
		return ml.layers();
	}

	@Override
	public Stream<ModuleLayer> layers(ClassLoader cl) {
		return ModuleLayer.layers(cl);
	}

	@Override
	public Module defineModule(ClassLoader cl, ModuleDescriptor md, URI uri) {
		return new Module(System.bootLayer, cl, md, uri);
	}

	@Override
	public Module defineUnnamedModule(ClassLoader cl) {
		return new Module(cl);
	}

	@Override
	public void addReads(Module fromModule, Module toModule) {
		fromModule.implAddReads(toModule);
	}

	@Override
	public void addReadsAllUnnamed(Module module) {
		module.implAddReadsAllUnnamed();
	}

	@Override
	public void addExports(Module fromModule, String pkg, Module toModule) {
		fromModule.implAddExports(pkg, toModule);
	}

	@Override
	public void addExportsToAllUnnamed(Module fromModule, String pkg) {
		fromModule.implAddExportsToAllUnnamed(pkg);
	}

	@Override
	public void addOpens(Module fromModule, String pkg, Module toModule) {
		fromModule.implAddOpens(pkg, toModule);
	}

	@Override
	public void addOpensToAllUnnamed(Module fromModule, String pkg) {
		fromModule.implAddOpensToAllUnnamed(pkg);
	}

	@Override
	public void addUses(Module fromModule, Class<?> service) {
		fromModule.implAddUses(service);
	}

	@Override
	public ServicesCatalog getServicesCatalog(ModuleLayer ml) {
		return ml.getServicesCatalog();
	}

	@SuppressWarnings("removal")
	@Override
	public void addNonExportedPackages(ModuleLayer ml) {
		SecurityManager.addNonExportedPackages(ml);
	}

	@Override
	public List<Method> getDeclaredPublicMethods(Class<?> clz, String name, Class<?>... types) {
		return clz.getDeclaredPublicMethods(name, types);
	}

	@Override
	public void addOpensToAllUnnamed(Module fromModule, Set<String> concealedPackages, Set<String> exportedPackages) {
		fromModule.implAddOpensToAllUnnamed(concealedPackages, exportedPackages);
	}

	@Override
	public boolean isReflectivelyOpened(Module fromModule, String pkg, Module toModule) {
		return fromModule.isReflectivelyOpened(pkg, toModule);
	}

	@Override
	public boolean isReflectivelyExported(Module fromModule, String pkg, Module toModule) {
		return fromModule.isReflectivelyExported(pkg, toModule);
	}

	@Override
	public String newStringUTF8NoRepl(byte[] bytes, int offset, int length) {
		return String.newStringUTF8NoRepl(bytes, offset, length, true);
	}

	@Override
	public byte[] getBytesUTF8NoRepl(String str) {
		return String.getBytesUTF8NoRepl(str);
	}

	@Override
	public void blockedOn(Interruptible interruptible) {
		Thread.blockedOn(interruptible);
	}

	@Override
	public byte[] getBytesNoRepl(String str, Charset charset) throws CharacterCodingException {
		return String.getBytesNoRepl(str, charset);
	}

	@Override
	public String newStringNoRepl(byte[] bytes, Charset charset) throws CharacterCodingException {
		return String.newStringNoRepl(bytes, charset);
	}

	@Override
	public void setCause(Throwable throwable, Throwable cause) {
		throwable.setCause(cause);
	}

	@Override
	public Class<?> defineClass(ClassLoader classLoader, Class<?> clazz, String className, byte[] classRep, ProtectionDomain protectionDomain, boolean init, int flags, Object classData) {
		ClassLoader targetClassLoader = (null == classLoader) ? ClassLoader.bootstrapClassLoader : classLoader;
		return targetClassLoader.defineClassInternal(clazz, className, classRep, protectionDomain, init, flags, classData);
	}

	/**
	 * Returns the classData stored in the class.
	 *
	 * @param the class from where to retrieve the classData.
	 *
	 * @return the classData (Object).
	 */
	@Override
	public Object classData(Class<?> clazz) {
		return clazz.getClassData();
	}

	@Override
	public ProtectionDomain protectionDomain(Class<?> clazz) {
		return clazz.getProtectionDomainInternal();
	}

	@Override
	public MethodHandle stringConcatHelper(String arg0, MethodType type) {
		return StringConcatHelper.lookupStatic(arg0, type);
	}

	@Override
	public long stringConcatInitialCoder() {
		return StringConcatHelper.initialCoder();
	}

	@Override
	public long stringConcatMix(long arg0, String string) {
		return StringConcatHelper.mix(arg0, string);
	}

	@Override
	public void bindToLoader(ModuleLayer ml, ClassLoader cl) {
		ml.bindToLoader(cl);
	}

	@Override
	public void addExports(Module fromModule, String pkg) {
		fromModule.implAddExports(pkg);
	}

	@Override
	public int decodeASCII(byte[] srcBytes, int srcPos, char[] dstChars, int dstPos, int length) {
		return String.decodeASCII(srcBytes, srcPos, dstChars, dstPos, length);
	}

	@Override
	public void inflateBytesToChars(byte[] srcBytes, int srcOffset, char[] dstChars, int dstOffset, int length) {
		StringLatin1.inflate(srcBytes, srcOffset, dstChars, dstOffset, length);
	}

	@Override
	public Class<?> findBootstrapClassOrNull(String name) {
		return VMAccess.findClassOrNull(name, ClassLoader.bootstrapClassLoader);
	}

	@Override
	public String join(String prefix, String suffix, String delimiter, String[] elements, int size) {
		return String.join(prefix, suffix, delimiter, elements, size);
	}

	@Override
	public void ensureNativeAccess(Module module, Class<?> ownerClass, String methodName) {
		module.ensureNativeAccess(ownerClass, methodName);
	}

	@Override
	public void addEnableNativeAccessToAllUnnamed() {
		Module.implAddEnableNativeAccessToAllUnnamed();
	}

	@Override
	public Module addEnableNativeAccess(Module mod) {
		return mod.implAddEnableNativeAccess();
	}

	@Override
	public boolean allowSecurityManager() {
		return System.allowSecurityManager();
	}

	@Override
	public long findNative(ClassLoader loader, String entryName) {
		return ClassLoader.findNative0(loader, entryName);
	}

	@Override
	public void exit(int status) {
		Shutdown.exit(status);
	}

	@Override
	public int encodeASCII(char[] sa, int sp, byte[] da, int dp, int len) {
		return StringCoding.implEncodeAsciiArray(sa, sp, da, dp, len);
	}

	@Override
	public Thread currentCarrierThread() {
		return Thread.currentCarrierThread();
	}

	@Override
	public <V> V executeOnCarrierThread(Callable<V> task) throws Exception {
		V result;
		Thread currentThread = Thread.currentThread();
		if (currentThread.isVirtual()) {
			Thread carrierThread = Thread.currentCarrierThread();
			carrierThread.setCurrentThread(carrierThread);
			try {
				result = task.call();
			} finally {
				carrierThread.setCurrentThread(currentThread);
			}
		} else {
			result = task.call();
		}
		return result;
	}

	@Override
	public Continuation getContinuation(Thread thread) {
		return thread.getContinuation();
	}

	@Override
	public void setContinuation(Thread thread, Continuation c) {
		thread.setContinuation(c);
	}

	@Override
	public Object[] scopedValueCache() {
		return Thread.scopedValueCache();
	}

	@Override
	public void setScopedValueCache(Object[] cache) {
		Thread.setScopedValueCache(cache);
	}

	@Override
	public Object scopedValueBindings() {
		return Thread.scopedValueBindings();
	}

	@Override
	public InputStream initialSystemIn() {
		return System.initialIn;
	}

	@Override
	public char getUTF16Char(byte[] val, int index) {
		return StringUTF16.getChar(val, index);
	}

	@Override
	public int countPositives(byte[] ba, int off, int len) {
		return StringCoding.countPositives(ba, off, len);
	}

	@Override
	public long stringConcatCoder(char value) {
		return StringConcatHelper.coder(value);
	}

	@Override
	public long stringBuilderConcatMix(long lengthCoder, StringBuilder sb) {
		return sb.mix(lengthCoder);
	}

	@Override
	public long stringBuilderConcatPrepend(long lengthCoder, byte[] buf, StringBuilder sb) {
		return sb.prepend(lengthCoder, buf);
	}

	@Override
	public StackableScope headStackableScope(Thread thread) {
		return thread.headStackableScopes();
	}

	@Override
	public void setHeadStackableScope(StackableScope scope) {
		Thread.setHeadStackableScope(scope);
	}

	@Override
	public ThreadContainer threadContainer(Thread thread) {
		return thread.threadContainer();
	}

	@Override
	public void start(Thread thread, ThreadContainer container) {
		thread.start(container);
	}

	@Override
	public Thread[] getAllThreads() {
		return Thread.getAllThreads();
	}

	@Override
	public ContinuationScope virtualThreadContinuationScope() {
		return VirtualThread.continuationScope();
	}

	@Override
	public void parkVirtualThread() {
		if (Thread.currentThread() instanceof BaseVirtualThread bvt) {
			bvt.park();
		} else {
			throw new WrongThreadException();
		}
	}

	@Override
	public void parkVirtualThread(long nanos) {
		if (Thread.currentThread() instanceof BaseVirtualThread bvt) {
			bvt.parkNanos(nanos);
		} else {
			throw new WrongThreadException();
		}
	}

	@Override
	public void unparkVirtualThread(Thread thread) {
		if (thread instanceof BaseVirtualThread bvt) {
			bvt.unpark();
		} else {
			throw new WrongThreadException();
		}
	}

	@Override
	public StackWalker newStackWalkerInstance(Set<StackWalker.Option> options, ContinuationScope contScope, Continuation continuation) {
		return StackWalker.newInstance(options, null, contScope, continuation);
	}

	/*
	 * To access package-private methods in ThreadLocal, an
	 * (implicit) cast from CarrierThreadLocal is required.
	 */
	private static <T> ThreadLocal<T> asThreadLocal(CarrierThreadLocal<T> local) {
		return local;
	}

	@Override
	public boolean isCarrierThreadLocalPresent(CarrierThreadLocal<?> carrierThreadlocal) {
		return asThreadLocal(carrierThreadlocal).isCarrierThreadLocalPresent();
	}

	@Override
	public <T> T getCarrierThreadLocal(CarrierThreadLocal<T> carrierThreadlocal) {
		return asThreadLocal(carrierThreadlocal).getCarrierThreadLocal();
	}

	@Override
	public void removeCarrierThreadLocal(CarrierThreadLocal<?> carrierThreadlocal) {
		asThreadLocal(carrierThreadlocal).removeCarrierThreadLocal();
	}

	@Override
	public <T> void setCarrierThreadLocal(CarrierThreadLocal<T> carrierThreadlocal, T carrierThreadLocalvalue) {
		asThreadLocal(carrierThreadlocal).setCarrierThreadLocal(carrierThreadLocalvalue);
	}

	@Override
	public String getLoaderNameID(ClassLoader loader) {
		if (loader == null) {
			return "null";
		}
		StringBuilder buffer = new StringBuilder();
		String name = loader.getName();

		if (name != null) {
			buffer.append("'").append(name).append("'");
		} else {
			buffer.append(loader.getClass().getName());
		}

		if (false == loader instanceof jdk.internal.loader.BuiltinClassLoader) {
			buffer.append(" @").append(Integer.toHexString(System.identityHashCode(loader)));
		}

		return buffer.toString();
	}

}
