/*
 * Copyright IBM Corp. and others 2009
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package com.ibm.j9ddr.corereaders.elf;

import java.io.IOException;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import com.ibm.j9ddr.corereaders.InvalidDumpFormatException;

/**
 * ELFDumpReader for IA32 dumps.
 * 
 * @author andhall
 *
 */
public class ELFIA32DumpReader extends ELFDumpReader
{

	ELFIA32DumpReader(ELFFileReader reader) throws IOException,
			InvalidDumpFormatException
	{
		super(reader);
	}

	/* (non-Javadoc)
	 * @see com.ibm.j9ddr.corereaders.elf.ELFDumpReader#readUID()
	 */
	@Override
	protected long readUID() throws IOException
	{
		return _reader.readShort() & 0xffffL;
	}

	@Override
	protected String getProcessorType()
	{
		return "x86";
	}

	@Override
	protected long getBasePointerFrom(Map<String, Number> registers)
	{
		return registers.get("ebp").longValue();
	}

	@Override
	protected long getInstructionPointerFrom(Map<String, Number> registers)
	{
		return registers.get("eip").longValue();
	}

	@Override
	protected long getLinkRegisterFrom(Map<String, Number> registers)
	{
		return 0;
	}

	@Override
	protected String getStackPointerRegisterName() {
		return "esp";
	}

	@Override
	protected SortedMap<String, Number> readRegisters() throws IOException
	{
		String[] names1 = { "ebx", "ecx", "edx", "esi", "edi", "ebp", "eax",
				"ds", "es", "fs", "gs" };
		String[] names2 = { "eip", "cs", "efl", "esp", "ss" };
		SortedMap<String, Number> registers = new TreeMap<String, Number>(new RegisterComparator());
		for (int i = 0; i < names1.length; i++)
			registers.put(names1[i], Long.valueOf(_reader.readInt() & 0xFFFFFFFFL));
		_reader.readInt(); // Ignore originalEax
		for (int i = 0; i < names2.length; i++)
			registers.put(names2[i], Long.valueOf(_reader.readInt() & 0xFFFFFFFFL));
		return registers;
	}

	@Override
	protected String[] getDwarfRegisterKeys() {
		// Needs to be overridden by each platform. Need to consult the platform ABI to map reg names to dwarf numbers.
		// (This list generated by looking at the order of the registers displayed by info registers in gdb)
		String[] registersVals = new String[16];
		registersVals[0] = "eax";
		registersVals[1] = "ecx";
		registersVals[2] = "edx";
		registersVals[3] = "ebx";
		registersVals[4] = "esp";
		registersVals[5] = "ebp";
		registersVals[6] = "esi";
		registersVals[7] = "edi";
		registersVals[8] = "eip";
		registersVals[9] = "eflags";
		registersVals[10] = "cs";
		registersVals[11] = "ss";
		registersVals[12] = "ds";
		registersVals[13] = "es";
		registersVals[14] = "fs";
		registersVals[15] = "gs";
		return registersVals;
	}
	
	@Override
	protected void readHighwordRegisters(DataEntry entry, Map<String, Number> registers)
		throws IOException, InvalidDumpFormatException {
			throw new InvalidDumpFormatException("Unexpected data entry in IA32 ELF dump");
	}

}
