/*
 * Copyright IBM Corp. and others 2012
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License 2.0 which accompanies this
 * distribution and is available at https://www.eclipse.org/legal/epl-2.0/
 * or the Apache License, Version 2.0 which accompanies this distribution and
 * is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * This Source Code may also be made available under the following
 * Secondary Licenses when the conditions for such availability set
 * forth in the Eclipse Public License, v. 2.0 are satisfied: GNU
 * General Public License, version 2 with the GNU Classpath
 * Exception [1] and GNU General Public License, version 2 with the
 * OpenJDK Assembly Exception [2].
 *
 * [1] https://www.gnu.org/software/classpath/license.html
 * [2] https://openjdk.org/legal/assembly-exception.html
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0 OR GPL-2.0-only WITH Classpath-exception-2.0 OR GPL-2.0-only WITH OpenJDK-assembly-exception-1.0
 */
package com.ibm.java.diagnostics.utils.plugins.impl;

import java.io.InputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Set;
import java.util.Optional;

import com.ibm.java.diagnostics.utils.plugins.Annotation;
import com.ibm.java.diagnostics.utils.plugins.ClassInfo;
import com.ibm.java.diagnostics.utils.plugins.ClassListener;

import java.lang.classfile.AnnotationElement;
import java.lang.classfile.AnnotationValue;
import java.lang.classfile.Attributes;
import java.lang.classfile.ClassFile;
import java.lang.classfile.ClassModel;
import java.lang.classfile.attribute.RuntimeVisibleAnnotationsAttribute;
import java.lang.classfile.constantpool.ClassEntry;
import java.util.function.Function;

public final class ClassScanner
{

	public static ClassInfo getClassInfo(InputStream file, URL url, Set<ClassListener> listeners) throws IOException {
		Function<ClassEntry, String> toDotted = entry -> entry.asInternalName().replace('/', '.');
		ClassModel model = ClassFile.of().parse(file.readAllBytes());
		int version = model.majorVersion() + (model.minorVersion() << 16);
		int access = model.flags().flagsMask();
		String dotName = toDotted.apply(model.thisClass());
		String signature = null; // not currently used
		String dotSuperName = model.superclass().map(toDotted).orElse(null);
		String[] interfaces = model.interfaces().stream().map(toDotted).toArray(String[]::new);

		ClassInfo info = new ClassInfo(dotName, url);

		for (String iface : interfaces) {
			info.addInterface(iface);
		}

		for (ClassListener listener : listeners) {
			listener.visit(version, access, dotName, signature, dotSuperName, interfaces);
		}

		Optional<RuntimeVisibleAnnotationsAttribute> attribute = model
				.findAttribute(Attributes.runtimeVisibleAnnotations());

		if (attribute.isPresent()) {
			for (java.lang.classfile.Annotation annotation : attribute.get().annotations()) {
				String classDescriptor = annotation.className().stringValue();
				Annotation current = info.addAnnotation(classDescriptor);

				for (ClassListener listener : listeners) {
					listener.visitAnnotation(classDescriptor, true /* visible */);
				}

				for (AnnotationElement element : annotation.elements()) {
					if (element.value() instanceof AnnotationValue.OfConstant constant) {
						current.addEntry(element.name().stringValue(), constant.resolvedValue());
					}
				}
			}
		}

		return info;
	}

	private ClassScanner() {
		super();
	}

}
